<?php

namespace App\Http\Controllers;

use App\Mail\ContactMail;
use App\Models\Contact;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class ContactController extends Controller
{
    /* ---------- Show form (English version) ---------- */
    public function showForm()
    {
        return view('contact'); // Blade file: resources/views/sections/contact-section.blade.php
    }

    /* ---------- Process form (English version) ---------- */
    public function processForm(Request $request)
    {
        // 1. Basic validation
        $validator = Validator::make(
            $request->all(),
            [
                'name'     => ['required', 'string', 'max:255'],
                'phone'    => ['required', 'string', 'max:20'],
                'email'    => ['required', 'email', 'max:255'],
                'comments' => ['required', 'string', 'max:1000'],
            ]
        );

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // 2. reCAPTCHA validation
        $recaptchaResponse = $request->input('g-recaptcha-response');

        if (!$recaptchaResponse) {
            return back()->withErrors(['g-recaptcha-response' => 'Please complete the reCAPTCHA'])->withInput();
        }

        $secretKey = env('RECAPTCHA_SECRET_KEY');
        $response = Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
            'secret'   => $secretKey,
            'response' => $recaptchaResponse,
            'remoteip' => $request->ip()
        ]);

        $body = $response->json();

        if (app()->environment('local')) {
            $body = ['success' => true];
        }

        if (!isset($body['success']) || $body['success'] !== true) {
            $errorMessage = 'reCAPTCHA verification failed.';

            if (isset($body['error-codes'])) {
                if (in_array('missing-input-secret', $body['error-codes'])) {
                    $errorMessage = 'The reCAPTCHA secret key is missing.';
                } elseif (in_array('invalid-input-secret', $body['error-codes'])) {
                    $errorMessage = 'Invalid reCAPTCHA secret key.';
                } elseif (in_array('timeout-or-duplicate', $body['error-codes'])) {
                    $errorMessage = 'reCAPTCHA token expired. Please try again.';
                }
            }

            return back()->withErrors(['g-recaptcha-response' => $errorMessage])->withInput();
        }

        // 3. Anti-spam filters (block Cyrillic characters)
        if (preg_match('/[А-Яа-яЁё]/u', $request->comments)) {
            return back()->withErrors(['comments' => 'Your message contains disallowed characters.'])->withInput();
        }

        $blockedWords = ['кредит', 'биткойн', 'инвестиции', 'привет'];
        foreach ($blockedWords as $word) {
            if (stripos($request->comments, $word) !== false) {
                return back()->withErrors(['comments' => 'Your message contains restricted content.'])->withInput();
            }
        }

        // 4. Process data (save to DB + send email)
        $data = $validator->validated();
        $data['fecha_hora'] = now()->setTimezone('America/Chicago')->format('l, F j, Y \a\t h:i A');

        try {
            Contact::create($data);

            // Send to multiple recipients (hardcoded)
            $recipients = [
                'jfenollosa@meypar-usa.com',
                'hturrent@meypar-usa.com',
                'fvaldovinos@eyspsa.com',
                
            ];

            Mail::to($recipients)->send(new ContactMail($data));

            return back()->with('success', 'Message sent! We will get back to you shortly.');
        } catch (\Exception $e) {
            Log::error('Error sending contact form (EN): ' . $e->getMessage());

            return back()->with(
                'error',
                'There was an error sending your message. Please try again. Error: ' . $e->getMessage()
            );
        }
    }
}
